/**
 * Flight Booking Application JavaScript
 * 
 * This file contains all the custom JavaScript for the flight booking system.
 * It includes functionality for search forms, booking flows, and interactive components.
 */

// Flight Search Enhancement
document.addEventListener('DOMContentLoaded', function() {
    
    // Auto-suggest for airport search
    function initAirportSearch() {
        const airportInputs = document.querySelectorAll('.airport-search');
        
        airportInputs.forEach(input => {
            input.addEventListener('input', function() {
                const query = this.value;
                if (query.length >= 2) {
                    // Here you would typically call an API to get airport suggestions
                    console.log('Searching for airports:', query);
                }
            });
        });
    }

    // Date picker enhancements
    function initDatePickers() {
        const dateInputs = document.querySelectorAll('input[type="date"]');
        
        dateInputs.forEach(input => {
            // Set minimum date to today
            const today = new Date().toISOString().split('T')[0];
            input.setAttribute('min', today);
            
            // Auto-fill return date when departure is selected
            if (input.id === 'departure-date') {
                input.addEventListener('change', function() {
                    const returnDateInput = document.getElementById('return-date');
                    if (returnDateInput && !returnDateInput.value) {
                        const departureDate = new Date(this.value);
                        departureDate.setDate(departureDate.getDate() + 7); // Default 7 days later
                        returnDateInput.value = departureDate.toISOString().split('T')[0];
                    }
                });
            }
        });
    }

    // Flight filter functionality
    function initFlightFilters() {
        const filterButtons = document.querySelectorAll('.filter-btn');
        const sortSelect = document.getElementById('sort-flights');
        
        filterButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Toggle active state
                this.classList.toggle('active');
                
                // Apply filter logic here
                console.log('Filter toggled:', this.dataset.filter);
            });
        });

        if (sortSelect) {
            sortSelect.addEventListener('change', function() {
                // Sort flights based on selection
                console.log('Sort by:', this.value);
            });
        }
    }

    // Booking form validation
    function initBookingValidation() {
        const bookingForm = document.getElementById('booking-form');
        
        if (bookingForm) {
            bookingForm.addEventListener('submit', function(e) {
                let isValid = true;
                
                // Validate required passenger information
                const requiredFields = this.querySelectorAll('[required]');
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.classList.add('border-red-500');
                    } else {
                        field.classList.remove('border-red-500');
                    }
                });
                
                // Validate email format
                const emailInputs = this.querySelectorAll('input[type="email"]');
                emailInputs.forEach(email => {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (email.value && !emailRegex.test(email.value)) {
                        isValid = false;
                        email.classList.add('border-red-500');
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                    alert('Please fill in all required fields correctly.');
                }
            });
        }
    }

    // Payment method selection
    function initPaymentMethods() {
        const paymentMethods = document.querySelectorAll('input[name="payment_method"]');
        
        paymentMethods.forEach(method => {
            method.addEventListener('change', function() {
                // Hide all payment forms
                document.querySelectorAll('.payment-form').forEach(form => {
                    form.style.display = 'none';
                });
                
                // Show selected payment form
                const selectedForm = document.getElementById(this.value + '-form');
                if (selectedForm) {
                    selectedForm.style.display = 'block';
                }
            });
        });
    }

    // Flight tracking
    function initFlightTracking() {
        const trackingForm = document.getElementById('flight-tracking-form');
        
        if (trackingForm) {
            trackingForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const flightNumber = document.getElementById('flight-number').value;
                if (flightNumber) {
                    // Make API call to track flight
                    fetch(`/api/flights/track/${flightNumber}`)
                        .then(response => response.json())
                        .then(data => {
                            displayFlightStatus(data);
                        })
                        .catch(error => {
                            console.error('Error tracking flight:', error);
                        });
                }
            });
        }
    }

    function displayFlightStatus(flightData) {
        const statusContainer = document.getElementById('flight-status');
        if (statusContainer) {
            statusContainer.innerHTML = `
                <div class="bg-white rounded-lg p-6 shadow-lg">
                    <h3 class="text-xl font-semibold mb-4">Flight Status</h3>
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <span class="text-sm text-slate-600">Flight Number:</span>
                            <div class="font-semibold">${flightData.flight_number}</div>
                        </div>
                        <div>
                            <span class="text-sm text-slate-600">Status:</span>
                            <div class="font-semibold ${getStatusColor(flightData.status)}">${flightData.status}</div>
                        </div>
                        <div>
                            <span class="text-sm text-slate-600">Departure:</span>
                            <div class="font-semibold">${flightData.departure_time}</div>
                        </div>
                        <div>
                            <span class="text-sm text-slate-600">Arrival:</span>
                            <div class="font-semibold">${flightData.arrival_time}</div>
                        </div>
                    </div>
                </div>
            `;
        }
    }

    function getStatusColor(status) {
        switch(status.toLowerCase()) {
            case 'on time':
                return 'text-green-600';
            case 'delayed':
                return 'text-yellow-600';
            case 'cancelled':
                return 'text-red-600';
            default:
                return 'text-slate-600';
        }
    }

    // Initialize all functionality
    initAirportSearch();
    initDatePickers();
    initFlightFilters();
    initBookingValidation();
    initPaymentMethods();
    initFlightTracking();
});

// Utility functions
window.FlightBooking = {
    // Format currency
    formatCurrency: function(amount, currency = 'USD') {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency
        }).format(amount);
    },

    // Format date
    formatDate: function(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    },

    // Format time
    formatTime: function(timeString) {
        const time = new Date('1970-01-01T' + timeString + 'Z');
        return time.toLocaleTimeString('en-US', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: true
        });
    },

    // Show loading spinner
    showLoading: function(element) {
        element.innerHTML = '<div class="flex items-center justify-center"><i class="fas fa-spinner fa-spin mr-2"></i> Loading...</div>';
    },

    // Show success message
    showSuccess: function(message) {
        this.showNotification(message, 'success');
    },

    // Show error message
    showError: function(message) {
        this.showNotification(message, 'error');
    },

    // Show notification
    showNotification: function(message, type = 'info') {
        const notification = document.createElement('div');
        const bgColor = type === 'success' ? 'bg-green-500' : type === 'error' ? 'bg-red-500' : 'bg-blue-500';
        
        notification.className = `fixed top-4 right-4 ${bgColor} text-white px-6 py-3 rounded-lg shadow-lg z-50 transform transition-transform duration-300 translate-x-full`;
        notification.innerHTML = `
            <div class="flex items-center">
                <span>${message}</span>
                <button class="ml-4 text-white hover:text-gray-200" onclick="this.parentElement.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.classList.remove('translate-x-full');
        }, 100);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 300);
        }, 5000);
    }
};