<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;

class FlightSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $airports = \App\Models\Airport::all();
        
        if ($airports->count() < 2) {
            $this->command->error('Please run AirportSeeder first!');
            return;
        }

        // Get first admin user or create a system user
        $adminUser = \App\Models\User::first();
        if (!$adminUser) {
            $this->command->info('No users found. Flights will be created without admin assignment.');
        }

        $airlines = [
            'American Airlines', 'Delta Air Lines', 'United Airlines', 'British Airways',
            'Air France', 'Lufthansa', 'Emirates', 'Qatar Airways', 'Singapore Airlines',
            'Japan Airlines', 'Cathay Pacific', 'KLM'
        ];

        $aircraftTypes = [
            'Boeing 737-800', 'Boeing 777-300ER', 'Boeing 787-9', 'Airbus A320',
            'Airbus A330-300', 'Airbus A350-900', 'Airbus A380-800'
        ];

        $statuses = ['scheduled', 'boarding', 'departed', 'airborne', 'landed'];

        // Create flights for the next 30 days
        for ($day = 0; $day < 30; $day++) {
            $date = now()->addDays($day);
            
            // Create 5-8 flights per day
            for ($i = 0; $i < rand(5, 8); $i++) {
                $originAirport = $airports->random();
                $destinationAirport = $airports->where('id', '!=', $originAirport->id)->random();
                
                $airline = $airlines[array_rand($airlines)];
                $flightNumber = $this->generateFlightNumber($airline);
                
                // Random departure time during the day
                $departureTime = $date->copy()->addHours(rand(6, 22))->addMinutes(rand(0, 59));
                
                // Flight duration between 1-12 hours
                $duration = rand(60, 720); // minutes
                $arrivalTime = $departureTime->copy()->addMinutes($duration);
                
                $seatsTotal = rand(150, 350);
                $seatsBooked = rand(0, min($seatsTotal, $seatsTotal * 0.9)); // Max 90% booked
                
                // Set status based on time
                $status = 'scheduled';
                if ($day === 0) { // Today
                    $now = now();
                    if ($departureTime->isPast()) {
                        if ($arrivalTime->isPast()) {
                            $status = 'landed';
                        } elseif ($departureTime->diffInHours($now) > 1) {
                            $status = rand(0, 1) ? 'airborne' : 'departed';
                        } else {
                            $status = 'boarding';
                        }
                    } elseif ($departureTime->diffInMinutes($now) <= 30) {
                        $status = 'boarding';
                    }
                }
                
                $flight = [
                    'airline' => $airline,
                    'flight_number' => $flightNumber,
                    'origin_airport_id' => $originAirport->id,
                    'destination_airport_id' => $destinationAirport->id,
                    'departure_time' => $departureTime,
                    'arrival_time' => $arrivalTime,
                    'status' => $status,
                    'terminal' => rand(1, 3),
                    'gate' => chr(rand(65, 71)) . rand(1, 20), // A1-G20
                    'aircraft_type' => $aircraftTypes[array_rand($aircraftTypes)],
                    'seats_total' => $seatsTotal,
                    'seats_available' => $seatsTotal - $seatsBooked,
                    'price' => rand(199, 1299) + 0.99, // $199.99 to $1299.99
                    'created_by' => $adminUser ? $adminUser->id : null,
                ];

                // Add tracking for in-flight aircraft
                if (in_array($status, ['airborne', 'departed'])) {
                    // Simulate position between origin and destination
                    $progress = rand(10, 90) / 100; // 10% to 90% progress
                    $flight['tracking_lat'] = $originAirport->latitude + 
                        ($destinationAirport->latitude - $originAirport->latitude) * $progress;
                    $flight['tracking_lng'] = $originAirport->longitude + 
                        ($destinationAirport->longitude - $originAirport->longitude) * $progress;
                    $flight['altitude'] = rand(30000, 42000);
                    $flight['speed'] = rand(450, 550);
                    $flight['last_position_updated_at'] = now()->subMinutes(rand(1, 30));
                }

                \App\Models\Flight::firstOrCreate(
                    [
                        'airline' => $flight['airline'],
                        'flight_number' => $flight['flight_number'],
                        'departure_time' => $flight['departure_time']
                    ],
                    $flight
                );
            }
        }

        $this->command->info('Seeded sample flights for 30 days');
    }

    private function generateFlightNumber($airline)
    {
        $prefixes = [
            'American Airlines' => 'AA',
            'Delta Air Lines' => 'DL',
            'United Airlines' => 'UA',
            'British Airways' => 'BA',
            'Air France' => 'AF',
            'Lufthansa' => 'LH',
            'Emirates' => 'EK',
            'Qatar Airways' => 'QR',
            'Singapore Airlines' => 'SQ',
            'Japan Airlines' => 'JL',
            'Cathay Pacific' => 'CX',
            'KLM' => 'KL'
        ];

        $prefix = $prefixes[$airline] ?? 'XX';
        return $prefix . rand(100, 9999);
    }
}
