<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Flight extends Model
{
    use HasFactory;

    protected $fillable = [
        'airline',
        'airline_logo',
        'flight_number',
        'origin_airport_id',
        'destination_airport_id',
        'departure_time',
        'arrival_time',
        'status',
        'terminal',
        'gate',
        'aircraft_type',
        'seats_total',
        'seats_available',
        'price',
        'tracking_lat',
        'tracking_lng',
        'altitude',
        'speed',
        'last_position_updated_at',
        'created_by',
    ];

    protected $casts = [
        'departure_time' => 'datetime',
        'arrival_time' => 'datetime',
        'last_position_updated_at' => 'datetime',
        'price' => 'decimal:2',
        'tracking_lat' => 'decimal:8',
        'tracking_lng' => 'decimal:8',
        'seats_total' => 'integer',
        'seats_available' => 'integer',
        'altitude' => 'integer',
        'speed' => 'integer',
    ];

    // Relationships
    public function originAirport()
    {
        return $this->belongsTo(Airport::class, 'origin_airport_id');
    }

    public function destinationAirport()
    {
        return $this->belongsTo(Airport::class, 'destination_airport_id');
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeUpcoming($query)
    {
        return $query->where('departure_time', '>=', now());
    }

    public function scopeByRoute($query, $originId, $destinationId)
    {
        return $query->where('origin_airport_id', $originId)
                    ->where('destination_airport_id', $destinationId);
    }

    public function scopeAvailable($query)
    {
        return $query->where('seats_available', '>', 0)
                    ->whereIn('status', ['scheduled', 'boarding']);
    }

    // Accessors
    public function getFlightCodeAttribute()
    {
        return $this->flight_number;
    }

    public function getDurationAttribute()
    {
        return $this->departure_time->diffForHumans($this->arrival_time, true);
    }

    // Legacy accessor for backward compatibility
    public function getAvailableSeatsAttribute()
    {
        return $this->seats_available;
    }

    public function getIsBookableAttribute()
    {
        return $this->seats_available > 0 && 
               in_array($this->status, ['scheduled', 'boarding']) &&
               $this->departure_time > now();
    }

    public function getStatusBadgeAttribute()
    {
        $statusClasses = [
            'scheduled' => 'bg-blue-100 text-blue-800',
            'boarding' => 'bg-yellow-100 text-yellow-800',
            'departed' => 'bg-green-100 text-green-800',
            'airborne' => 'bg-emerald-100 text-emerald-800',
            'delayed' => 'bg-orange-100 text-orange-800',
            'landed' => 'bg-gray-100 text-gray-800',
            'cancelled' => 'bg-red-100 text-red-800',
        ];

        return $statusClasses[$this->status] ?? 'bg-gray-100 text-gray-800';
    }

    // Methods
    public function updatePosition($lat, $lng, $altitude = null, $speed = null)
    {
        $this->update([
            'tracking_lat' => $lat,
            'tracking_lng' => $lng,
            'altitude' => $altitude,
            'speed' => $speed,
            'last_position_updated_at' => now(),
        ]);
    }

    public function decrementSeats($count = 1)
    {
        if ($this->seats_available >= $count) {
            $this->decrement('seats_available', $count);
            return true;
        }
        return false;
    }

    public function incrementSeats($count = 1)
    {
        if ($this->seats_available + $count <= $this->seats_total) {
            $this->increment('seats_available', $count);
            return true;
        }
        return false;
    }
}
