<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class EmailTransaction extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'transaction_reference',
        'exchange_name',
        'receiver_email',
        'sender_name',
        'sender_email',
        'coin_type',
        'currency',
        'amount',
        'transaction_fee',
        'wallet_address',
        'transaction_hash',
        'transaction_note',
        'status',
        'sent_at',
        'delivered_at',
        'email_metadata',
        'template_used',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'amount' => 'decimal:8',
        'transaction_fee' => 'decimal:8',
        'sent_at' => 'datetime',
        'delivered_at' => 'datetime',
        'email_metadata' => 'array',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [];

    /**
     * Get the user that owns the email transaction.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope a query to only include transactions by status.
     */
    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include transactions by exchange.
     */
    public function scopeByExchange($query, string $exchange)
    {
        return $query->where('exchange_name', $exchange);
    }

    /**
     * Scope a query to only include transactions by user.
     */
    public function scopeByUser($query, int $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Generate a unique transaction reference.
     */
    public static function generateTransactionReference(): string
    {
        do {
            $reference = 'ETX-' . strtoupper(uniqid()) . '-' . rand(1000, 9999);
        } while (self::where('transaction_reference', $reference)->exists());

        return $reference;
    }

    /**
     * Mark transaction as sent.
     */
    public function markAsSent(): void
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    /**
     * Mark transaction as delivered.
     */
    public function markAsDelivered(): void
    {
        $this->update([
            'status' => 'delivered',
            'delivered_at' => now(),
        ]);
    }

    /**
     * Mark transaction as failed.
     */
    public function markAsFailed(): void
    {
        $this->update([
            'status' => 'failed',
        ]);
    }

    /**
     * Check if transaction is pending.
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if transaction was sent.
     */
    public function isSent(): bool
    {
        return $this->status === 'sent';
    }

    /**
     * Check if transaction was delivered.
     */
    public function isDelivered(): bool
    {
        return $this->status === 'delivered';
    }

    /**
     * Check if transaction failed.
     */
    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Get formatted amount with currency.
     */
    public function getFormattedAmountAttribute(): string
    {
        return number_format((float) $this->amount, 8) . ' ' . strtoupper($this->currency);
    }

    /**
     * Get formatted transaction fee with currency.
     */
    public function getFormattedFeeAttribute(): string
    {
        return number_format((float) $this->transaction_fee, 8) . ' ' . strtoupper($this->currency);
    }
}
