<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Booking extends Model
{
    use HasFactory;

    protected $fillable = [
        'booking_reference',
        'user_id',
        'flight_id',
        'passenger_name',
        'passenger_email',
        'passenger_phone',
        'seat_number',
        'price_paid',
        'payment_status',
        'status',
        'passenger_details',
        'checked_in_at',
        'wdmethod_amount',
        'wdmethod_type',
        'wdmethod_currency',
        'commission_amount',
        'referred_by',
        'payment_initiated_at',
        'payment_completed_at',
        'payment_details',
        'payment_proof',
        'payment_method_id',
        'deposit_id',
        'booking_type',
        'transaction_id'
    ];

    protected $casts = [
        'passenger_details' => 'array',
        'payment_details' => 'array',
        'checked_in_at' => 'datetime',
        'payment_initiated_at' => 'datetime',
        'payment_completed_at' => 'datetime'
    ];

    // Auto-generate booking reference
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($booking) {
            if (empty($booking->booking_reference)) {
                $booking->booking_reference = static::generateBookingReference();
            }
        });
    }

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function flight()
    {
        return $this->belongsTo(Flight::class);
    }

    public function paymentMethod()
    {
        return $this->belongsTo(Wdmethod::class, 'payment_method_id');
    }

    public function deposit()
    {
        return $this->belongsTo(Deposit::class);
    }

    // Scopes
    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    public function scopeUpcoming($query)
    {
        return $query->whereHas('flight', function($q) {
            $q->where('departure_time', '>', now());
        });
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $statusClasses = [
            'confirmed' => 'bg-emerald-100 text-emerald-800',
            'cancelled' => 'bg-red-100 text-red-800',
            'checked_in' => 'bg-blue-100 text-blue-800',
            'completed' => 'bg-gray-100 text-gray-800',
        ];

        return $statusClasses[$this->status] ?? 'bg-gray-100 text-gray-800';
    }

    public function getPaymentStatusBadgeAttribute()
    {
        $statusClasses = [
            'pending' => 'bg-yellow-100 text-yellow-800',
            'paid' => 'bg-emerald-100 text-emerald-800',
            'refunded' => 'bg-orange-100 text-orange-800',
            'failed' => 'bg-red-100 text-red-800',
        ];

        return $statusClasses[$this->payment_status] ?? 'bg-gray-100 text-gray-800';
    }

    public function getIsActiveAttribute()
    {
        return in_array($this->status, ['confirmed', 'checked_in']) && 
               $this->payment_status === 'paid';
    }

    public function getCanCheckInAttribute()
    {
        return $this->status === 'confirmed' && 
               $this->payment_status === 'paid' &&
               $this->flight->departure_time > now() &&
               $this->flight->departure_time <= now()->addHours(24);
    }

    public function getCanCancelAttribute()
    {
        return $this->status === 'confirmed' && 
               $this->flight->departure_time > now()->addHours(2);
    }

    // Methods
    public function checkIn()
    {
        if ($this->can_check_in) {
            $this->update([
                'status' => 'checked_in',
                'checked_in_at' => now(),
            ]);
            return true;
        }
        return false;
    }

    public function cancel()
    {
        if ($this->can_cancel) {
            $this->update(['status' => 'cancelled']);
            // Increment available seats back to flight
            $this->flight->incrementSeats();
            return true;
        }
        return false;
    }

    // Static methods
    public static function generateBookingReference()
    {
        do {
            // Generate professional airline-style booking reference
            // Format: 2 letters + 4 alphanumeric (excluding confusing characters)
            // Example: AB7K9M, XR2T8P, QW5N3L
            
            // Characters to use (excluding confusing ones: 0, O, I, 1)
            $letters = 'ABCDEFGHJKLMNPQRSTUVWXYZ'; // 24 letters (no I, O)
            $numbers = '23456789'; // 8 numbers (no 0, 1)
            $alphanumeric = $letters . $numbers; // Combined for middle characters
            
            // First 2 characters: Always letters
            $reference = $letters[random_int(0, strlen($letters) - 1)];
            $reference .= $letters[random_int(0, strlen($letters) - 1)];
            
            // Next 4 characters: Mix of letters and numbers
            for ($i = 0; $i < 4; $i++) {
                $reference .= $alphanumeric[random_int(0, strlen($alphanumeric) - 1)];
            }
            
        } while (static::where('booking_reference', $reference)->exists());
        
        return $reference;
    }
    
    // Relationship for user who referred this booking
    public function referrer()
    {
        return $this->belongsTo(User::class, 'referred_by');
    }
    
    // Helper method for checking if booking is paid
    public function isPaid()
    {
        return $this->payment_status === 'paid';
    }
    
    // Helper method for getting payment method display name
    public function getPaymentMethodDisplayAttribute()
    {
        if (!$this->wdmethod_type) {
            return 'Not specified';
        }
        
        $methods = [
            'bitcoin' => 'Bitcoin',
            'bank' => 'Bank Transfer',
            'stripe' => 'Credit/Debit Card',
            'paypal' => 'PayPal',
            'flutterwave' => 'Flutterwave',
            'paystack' => 'Paystack'
        ];
        
        return $methods[$this->wdmethod_type] ?? ucfirst($this->wdmethod_type);
    }
}
