<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class FlashEmailRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true; // Allow all users for now, add authentication logic if needed
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'exchange' => 'required|string|max:50',
            'coin' => 'required|string|max:10',
            'amount' => 'required|numeric|min:0.001|max:100000',
            'currency' => 'nullable|string|max:10',
            'quantity' => 'nullable|numeric|min:0',
            'receiverAddress' => 'required|string|max:255',
            'transactionFee' => 'nullable|numeric|min:0',
            'senderEmail' => 'required|email|max:255',
            'receiverEmail' => 'required|email|max:255',
            'senderPhone' => 'nullable|string|max:20',
            'receiverPhone' => 'nullable|string|max:20',
            'senderName' => 'nullable|string|max:100',
            'receiverName' => 'nullable|string|max:100',
            'chaintype' => 'nullable|string|max:50',
            'warningMessage' => 'nullable|string|max:500',
             'emailType' => 'required|in:withdrawal,deposit',
            
        ];
    }

    /**
     * Get custom error messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'exchange.required' => 'Please select an exchange.',
            'coin.required' => 'Please select a cryptocurrency.',
            'amount.required' => 'Amount is required.',
            'amount.numeric' => 'Amount must be a valid number.',
            'amount.min' => 'Amount must be at least 0.001.',
            'amount.max' => 'Amount cannot exceed 100,000.',
            'receiverAddress.required' => 'Receiver wallet address is required.',
            'receiverAddress.max' => 'Receiver address is too long.',
            'senderEmail.required' => 'Sender email is required.',
            'senderEmail.email' => 'Please enter a valid sender email address.',
            'receiverEmail.required' => 'Receiver email is required.',
            'receiverEmail.email' => 'Please enter a valid receiver email address.',
            'warningMessage.max' => 'Warning message cannot exceed 500 characters.',
             'emailType.required' => 'Please choose whether this is a deposit or withdrawal.',
             
        ];
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        $this->merge([
            'amount' => abs((float) $this->amount),
            'quantity' => abs((float) ($this->quantity ?? 0)),
            'transactionFee' => abs((float) ($this->transactionFee ?? 0)),
            'senderEmail' => filter_var($this->senderEmail, FILTER_SANITIZE_EMAIL),
            'receiverEmail' => filter_var($this->receiverEmail, FILTER_SANITIZE_EMAIL),
        ]);
    }

    /**
     * Get the validated data with defaults for optional fields.
     */
    public function getValidatedDataWithDefaults(): array
    {
        $validated = $this->validated();

        return array_merge([
            'currency' => $validated['coin'],
            'quantity' => $validated['amount'],
            'transactionFee' => 0,
            'senderPhone' => '',
            'receiverPhone' => '',
            'senderName' => 'Sender',
            'receiverName' => 'Receiver',
            'chaintype' => '',
            'warningMessage' => '',
             'emailType' => $this->input('emailType'),
        ], $validated);
    }
}
