<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Flight;
use App\Models\Airport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class FlightSearchController extends Controller
{
    public function index()
    {
        // Get all airports for search form
        $airports = Airport::orderBy('city')->get();
        
        // Get available flights for today and tomorrow
        $availableFlights = Flight::with(['originAirport', 'destinationAirport'])
            ->where('departure_time', '>=', now())
            ->where('departure_time', '<=', now()->addDays(3))
            ->where('seats_available', '>', 0)
            ->whereIn('status', ['scheduled', 'boarding'])
            ->orderBy('departure_time', 'asc')
            ->limit(8)
            ->get();
        
        // Get popular destinations (only the 6 with images)
        $popularDestinationCodes = ['JFK', 'LAX', 'CDG', 'LHR', 'MIA', 'SYD'];
        $popularDestinations = Airport::whereIn('iata_code', $popularDestinationCodes)
            ->get()
            ->sortBy(function($airport) use ($popularDestinationCodes) {
                return array_search($airport->iata_code, $popularDestinationCodes);
            })
            ->values()
            ->map(function ($airport) {
                $airport->min_price = Flight::where('destination_airport_id', $airport->id)
                    ->where('departure_time', '>', now())
                    ->min('price') ?? 299;
                return $airport;
            });
        
        return view('user.flights.search', compact('airports', 'popularDestinations', 'availableFlights'));
    }
    
    public function search(Request $request)
    {
        $validated = $request->validate([
            'origin' => 'required|exists:airports,iata_code',
            'destination' => 'required|exists:airports,iata_code',
            'departure_date' => 'required|date|after_or_equal:today',
            'return_date' => 'nullable|date|after:departure_date',
            'passengers' => 'required|integer|min:1|max:9',
            'class' => 'nullable|in:economy,business,first',
            'sort' => 'nullable|in:price,departure,duration,airline',
            'max_price' => 'nullable|numeric|min:0',
            'times' => 'nullable|string',
            'stops' => 'nullable|in:direct,any',
            'airlines' => 'nullable|string'
        ]);
        
        // Get airports for the search
        $originAirport = Airport::where('iata_code', $validated['origin'])->first();
        $destinationAirport = Airport::where('iata_code', $validated['destination'])->first();
        
        // Build flights query
        $query = Flight::with(['originAirport', 'destinationAirport'])
            ->where('origin_airport_id', $originAirport->id)
            ->where('destination_airport_id', $destinationAirport->id)
            ->whereDate('departure_time', $validated['departure_date'])
            ->where('seats_available', '>=', $validated['passengers'])
            ->where('departure_time', '>', now());
        
        // Apply filters
        if (!empty($validated['max_price'])) {
            $query->where('price', '<=', $validated['max_price']);
        }
        
        if (!empty($validated['times'])) {
            $timeFilters = explode(',', $validated['times']);
            $query->where(function($q) use ($timeFilters) {
                foreach ($timeFilters as $time) {
                    switch ($time) {
                        case 'morning':
                            $q->orWhereBetween(DB::raw('HOUR(departure_time)'), [6, 11]);
                            break;
                        case 'afternoon':
                            $q->orWhereBetween(DB::raw('HOUR(departure_time)'), [12, 17]);
                            break;
                        case 'evening':
                            $q->orWhereBetween(DB::raw('HOUR(departure_time)'), [18, 23]);
                            break;
                        case 'night':
                            $q->orWhereBetween(DB::raw('HOUR(departure_time)'), [0, 5]);
                            break;
                    }
                }
            });
        }
        
        if (!empty($validated['airlines'])) {
            $airlines = explode(',', $validated['airlines']);
            $query->whereIn('airline', $airlines);
        }
        
        // Apply sorting
        switch ($validated['sort'] ?? 'price') {
            case 'price':
                $query->orderBy('price', 'asc');
                break;
            case 'departure':
                $query->orderBy('departure_time', 'asc');
                break;
            case 'duration':
                $query->orderBy('duration', 'asc');
                break;
            case 'airline':
                $query->orderBy('airline', 'asc');
                break;
            default:
                $query->orderBy('price', 'asc');
        }
        
        // Get flights with pagination
        $flights = $query->paginate(10);
        
        // Get maximum price for filters
        $maxPrice = Flight::where('origin_airport_id', $originAirport->id)
            ->where('destination_airport_id', $destinationAirport->id)
            ->whereDate('departure_time', $validated['departure_date'])
            ->where('departure_time', '>', now())
            ->max('price') ?? 1000;
        
        // Get unique airlines for filters
        $airlines = Flight::where('origin_airport_id', $originAirport->id)
            ->where('destination_airport_id', $destinationAirport->id)
            ->whereDate('departure_time', $validated['departure_date'])
            ->where('departure_time', '>', now())
            ->distinct()
            ->pluck('airline')
            ->sort()
            ->values();
        
        return view('user.flights.results', compact('flights', 'validated', 'maxPrice', 'airlines'))
               ->with('searchParams', $validated);
    }
    
    // Get flight details for tracking
    public function track(Request $request)
    {
        $flightNumber = $request->input('flight_number');
        $date = $request->input('date', now()->format('Y-m-d'));
        
        if (!$flightNumber) {
            return response()->json(['error' => 'Flight number is required'], 400);
        }
        
        $flight = Flight::with(['originAirport', 'destinationAirport'])
            ->where('flight_number', $flightNumber)
            ->whereDate('departure_time', $date)
            ->first();
        
        if (!$flight) {
            return response()->json(['error' => 'Flight not found'], 404);
        }
        
        // Calculate progress percentage
        $totalDuration = $flight->departure_time->diffInMinutes($flight->arrival_time);
        $elapsed = now()->diffInMinutes($flight->departure_time, false);
        $progress = max(0, min(100, ($elapsed / $totalDuration) * 100));
        
        return response()->json([
            'flight' => [
                'number' => $flight->flight_number,
                'airline' => $flight->airline,
                'status' => $flight->status,
                'departure' => [
                    'airport' => $flight->originAirport->name,
                    'iata' => $flight->originAirport->iata_code,
                    'city' => $flight->originAirport->city,
                    'time' => $flight->departure_time->format('H:i'),
                    'date' => $flight->departure_time->format('M d, Y')
                ],
                'arrival' => [
                    'airport' => $flight->destinationAirport->name,
                    'iata' => $flight->destinationAirport->iata_code,
                    'city' => $flight->destinationAirport->city,
                    'time' => $flight->arrival_time->format('H:i'),
                    'date' => $flight->arrival_time->format('M d, Y')
                ],
                'duration' => $flight->duration,
                'current_position' => $flight->current_position,
                'progress' => round($progress, 1)
            ]
        ]);
    }
}
