<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Flight;
use App\Models\Booking;
use App\Models\Wdmethod;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class FlightBookingController extends Controller
{
    // Show flight details and booking form
    public function show(Flight $flight)
    {
        // Load flight with relationships
        $flight->load(['originAirport', 'destinationAirport']);
        
        // Get available payment methods for deposits (same as deposit page)
        $paymentMethods = Wdmethod::where(function ($query) {
            $query->where('type', '=', 'deposit')
                ->orWhere('type', '=', 'both');
        })->where('status', 'enabled')->orderByDesc('id')->get();
        
        return view('user.flights.booking', compact('flight', 'paymentMethods'));
    }
    
    // Process booking creation and payment
    public function store(Request $request, Flight $flight)
    {
        // Validate request
        $validated = $request->validate([
            'passenger_name' => 'required|string|max:255',
            'passenger_email' => 'required|email|max:255',
            'passenger_phone' => 'nullable|string|max:20',
            'passenger_dob' => 'nullable|date',
            'nationality' => 'nullable|string|max:100',
            'passport_number' => 'nullable|string|max:50',
            'special_requests' => 'nullable|string|max:500',
            'payment_method' => 'required|exists:wdmethods,id',
            'referral_code' => 'nullable|string|exists:users,referral_code'
        ]);
        
        // Refresh flight data to get latest seat count
        $flight->refresh();
        
        // Check flight availability
        if ($flight->seats_available <= 0) {
            return redirect()->back()->with('error', 'This flight is fully booked.');
        }
        
        // Check if flight is bookable
        if (!$flight->is_bookable) {
            return redirect()->back()->with('error', 'This flight is no longer available for booking.');
        }
        
        // Get payment method
        $wdmethod = Wdmethod::findOrFail($validated['payment_method']);
        
        // Calculate commission if referral code provided
        $referredBy = null;
        $commissionAmount = 0;
        if (!empty($validated['referral_code'])) {
            $referrer = User::where('referral_code', $validated['referral_code'])->first();
            if ($referrer) {
                $referredBy = $referrer->id;
                // Calculate 5% commission (you can adjust this)
                $commissionAmount = $flight->price * 0.05;
            }
        }
        
        try {
            DB::beginTransaction();
            
            // Double-check availability within transaction to prevent race conditions
            $currentFlight = Flight::lockForUpdate()->find($flight->id);
            if ($currentFlight->seats_available <= 0) {
                DB::rollBack();
                return redirect()->back()->with('error', 'This flight is fully booked.');
            }
            
            // Create booking
            $booking = Booking::create([
                'booking_reference' => Booking::generateBookingReference(),
                'user_id' => Auth::id(), // null if guest
                'flight_id' => $flight->id,
                'passenger_name' => $validated['passenger_name'],
                'passenger_email' => $validated['passenger_email'],
                'passenger_phone' => $validated['passenger_phone'],
                'price_paid' => $currentFlight->price * 1.1, // Include taxes and fees
                'payment_status' => 'pending',
                'status' => 'confirmed',
                'payment_method_id' => $wdmethod->id, // Store payment method ID
                'booking_type' => Auth::check() ? 'user' : 'guest', // Track booking type
                'wdmethod_amount' => $currentFlight->price * 1.1,
                'wdmethod_type' => $wdmethod->name,
                'wdmethod_currency' => 'USD', // Assuming USD for flights
                'commission_amount' => $commissionAmount,
                'referred_by' => $referredBy,
                'payment_initiated_at' => now(),
                'passenger_details' => [
                    'dob' => $validated['passenger_dob'] ?? null,
                    'nationality' => $validated['nationality'] ?? null,
                    'passport_number' => $validated['passport_number'] ?? null,
                    'special_requests' => $validated['special_requests'] ?? null,
                ],
                'payment_details' => [
                    'method_id' => $wdmethod->id,
                    'method_name' => $wdmethod->name,
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]
            ]);
            
            // Decrement available seats
            $decrementResult = $currentFlight->decrementSeats();
            if (!$decrementResult) {
                DB::rollBack();
                return redirect()->back()->with('error', 'Failed to reserve seat. Please try again.');
            }
            
            DB::commit();
            
            // Redirect to payment processing
            return redirect()->route('flights.booking.payment', $booking)
                           ->with('success', 'Booking created successfully. Please complete payment.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Flight booking failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Booking failed. Please try again.');
        }
    }
    
    // Show payment page
    public function payment(Booking $booking)
    {
        // Ensure user can access this booking
        if (Auth::check() && $booking->user_id !== Auth::id()) {
            abort(403);
        }
        
        // Check if already paid
        if ($booking->isPaid()) {
            return redirect()->route('flights.booking.success', $booking);
        }
        
        // Load flight with relationships
        $booking->load(['flight.originAirport', 'flight.destinationAirport']);
        
        // Get payment method details
        $wdmethod = Wdmethod::where('name', $booking->wdmethod_type)->first();
        
        // If payment method is not found, get a default one or show error
        if (!$wdmethod) {
            return redirect()->route('flights.booking', $booking->flight)
                           ->with('error', 'Payment method not found. Please try booking again.');
        }
        
        return view('user.flights.payment', compact('booking', 'wdmethod'));
    }
    
    // Handle payment confirmation
    public function confirmPayment(Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'payment_proof' => 'nullable|image|max:2048', // For bank transfer proof
            'transaction_id' => 'nullable|string|max:255'
        ]);
        
        try {
            // Handle payment proof upload
            $paymentProofPath = null;
            if ($request->hasFile('payment_proof')) {
                $paymentProofPath = $request->file('payment_proof')->store('payment-proofs', 'public');
            }
            
            // Update payment status based on payment method
            $paymentDetails = $booking->payment_details ?? [];
            $paymentDetails['confirmation_time'] = now()->toISOString();
            $paymentDetails['transaction_id'] = $validated['transaction_id'] ?? null;
            
            // For automatic methods (Stripe, PayPal), mark as paid immediately
            // For manual methods (Bitcoin, Bank Transfer), mark as pending review
            if (in_array($booking->wdmethod_type, ['Stripe', 'PayPal', 'stripe', 'paypal'])) {
                $booking->update([
                    'payment_status' => 'paid',
                    'payment_completed_at' => now(),
                    'payment_details' => $paymentDetails,
                    'transaction_id' => $validated['transaction_id'] ?? null,
                    'payment_proof' => $paymentProofPath
                ]);
                
                // Credit commission if applicable
                if ($booking->referred_by && $booking->commission_amount > 0) {
                    $this->creditReferralCommission($booking);
                }
            } else {
                // For manual payment methods (Bitcoin, Bank Transfer, etc.)
                // Keep status as pending until admin approves
                $booking->update([
                    'payment_details' => $paymentDetails,
                    'transaction_id' => $validated['transaction_id'] ?? null,
                    'payment_proof' => $paymentProofPath
                ]);
            }
            
            return redirect()->route('flights.booking.success', $booking);
            
        } catch (\Exception $e) {
            Log::error('Payment confirmation failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Payment confirmation failed. Please try again.');
        }
    }
    
    // Show booking success page
    public function success(Booking $booking)
    {
        $booking->load(['flight.originAirport', 'flight.destinationAirport', 'paymentMethod']);
        
        return view('user.flights.success', compact('booking'));
    }
    
    // Show user's bookings
    public function myBookings(Request $request)
    {
        // If user is authenticated, show their bookings
        if (Auth::check()) {
            $bookings = Booking::with(['flight.originAirport', 'flight.destinationAirport'])
                ->where('user_id', Auth::id())
                ->orderBy('created_at', 'desc')
                ->paginate(10);
            
            return view('user.flights.my-bookings', compact('bookings'));
        }
        
        // If guest, show search form to find booking by reference
        return view('user.flights.my-bookings', ['bookings' => collect()]);
    }
    
    // Show booking details by reference (for guests)
    public function showByReference($reference)
    {
        $booking = Booking::with(['flight.originAirport', 'flight.destinationAirport', 'paymentMethod'])
            ->where('booking_reference', strtoupper($reference))
            ->firstOrFail();
        
        return view('user.flights.booking-details', compact('booking'));
    }
    
    // Credit referral commission
    private function creditReferralCommission(Booking $booking)
    {
        if ($booking->referred_by && $booking->commission_amount > 0) {
            $referrer = User::find($booking->referred_by);
            if ($referrer) {
                // Add commission to referrer's account balance
                // This follows the existing deposit pattern
                $referrer->increment('account_bal', $booking->commission_amount);
                
                // You might want to create a transaction record here
                // following your existing transaction table structure
            }
        }
    }
}
