<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\FlashEmailRequest;
use App\Models\EmailTransaction;
use App\Models\Settings;
use Carbon\Carbon;

class FlashServiceController extends Controller
{
    /**
     * Display the flash service form
     */
    public function index()
    {
        $title = 'Flash Service - Generate Email Transaction';
        return view('user.flashservice', compact('title'));
    }

    /**
     * Process the flash email transaction
     */
    public function processTransaction(FlashEmailRequest $request)
    {
        $user = Auth::user();
        $settings = Settings::first();
        $flashPrice = $settings->flash_price ?? 0;

        if ($flashPrice > 0 && $user->account_bal < $flashPrice) {
            return back()->with('message', 'Insufficient balance to perform this transaction. Please fund your account.')->with('status', 'error');
        }

        try {
            // Deduct the flash price from user's balance if there is a price
            if ($flashPrice > 0) {
                $user->account_bal -= $flashPrice;
                $user->save();
            }

            // Extract validated data with defaults
            $validatedData = $request->getValidatedDataWithDefaults();

            // DEBUG: Check what's happening with emailType
            Log::info('EmailType Debug - Form submission', [
                'raw_emailType_from_request' => $request->input('emailType'),
                'validated_data_emailType' => $validatedData['emailType'] ?? 'NOT_SET',
                'validated_data_keys' => array_keys($validatedData),
                'all_form_data' => $request->all(),
                'validated_data_full' => $validatedData
            ]);

            // Generate timestamp and transaction hash
            $timestamp = Carbon::now()->format('Y-m-d H:i:s');
            $transactionHash = bin2hex(random_bytes(32));
            $transactionReference = EmailTransaction::generateTransactionReference();

            // Add generated fields to data
            $emailData = array_merge($validatedData, [
                'timestamp' => $timestamp,
                'transactionHash' => $transactionHash,
                'transactionReference' => $transactionReference,
                'senderName' => $validatedData['senderName'] ?? 'Sender',
                'receiverName' => $validatedData['receiverName'] ?? 'Receiver',
            ]);

            // Create initial transaction record
            $transaction = EmailTransaction::create([
                'user_id' => Auth::id(),
                'transaction_reference' => $transactionReference,
                'exchange_name' => $emailData['exchange'],
                'receiver_email' => $emailData['receiverEmail'],
                'sender_name' => $emailData['senderName'],
                'sender_email' => $emailData['senderEmail'],
                'coin_type' => $emailData['coin'],
                'currency' => $emailData['currency'],
                'amount' => (float) $emailData['amount'],
                'transaction_fee' => (float)$settings->flash_price,
                'wallet_address' => $emailData['receiverAddress'],
                'transaction_hash' => $transactionHash,
                'transaction_note' => $emailData['warningMessage'] ?? null,
                'status' => 'pending',
                'email_metadata' => [
                    'quantity' => $emailData['quantity'],
                    'chaintype' => $emailData['chaintype'],
                    'sender_phone' => $emailData['senderPhone'],
                    'receiver_phone' => $emailData['receiverPhone'],
                    'timestamp' => $timestamp,
                    'email_type' => $emailData['emailType'] ?? 'withdrawal',
                ],
            ]);

            // Define email subjects for different exchanges
            $emailSubjects = $this->getEmailSubjects($emailData);

            // Get the selected email type from the form - use multiple fallbacks
            $selectedEmailType = $emailData['emailType'] ?? $request->input('emailType') ?? 'withdrawal';

            // Debug: Log the email type decision process
            Log::info('EmailType Debug - Selection process', [
                'emailData_emailType' => $emailData['emailType'] ?? 'NOT_SET',
                'request_input_emailType' => $request->input('emailType'),
                'final_selectedEmailType' => $selectedEmailType,
                'transaction_reference' => $transactionReference
            ]);

            // Define template types and their corresponding recipient emails based on user selection
            $templates = [];
            if ($selectedEmailType === 'withdrawal') {
                // Send withdrawal email to sender AND a copy to receiver
                $templates['withdrawal'] = [
                    ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']],
                    ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']]
                ];
                Log::info('Will send WITHDRAWAL email to: ' . $emailData['senderEmail'] . ' and copy to: ' . $emailData['receiverEmail']);
            } elseif ($selectedEmailType === 'deposit') {
                // Send deposit email to receiver AND a copy to sender
                $templates['deposit'] = [
                    ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']],
                    ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']]
                ];
                Log::info('Will send DEPOSIT email to: ' . $emailData['receiverEmail'] . ' and copy to: ' . $emailData['senderEmail']);
            } else {
                // Fallback to both if somehow an invalid type is selected
                Log::warning('Invalid email type, sending both emails', ['selectedEmailType' => $selectedEmailType]);
                $templates = [
                    'withdrawal' => [
                        ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']],
                        ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']]
                    ],
                    'deposit' => [
                        ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']],
                        ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']]
                    ]
                ];
            }

            $emailErrors = [];
            $emailsSent = 0;
            $templatesUsed = [];

            foreach ($templates as $templateType => $recipientData) {
                // Handle both single recipient and multiple recipients (array)
                $recipients = isset($recipientData[0]) && is_array($recipientData[0]) ? $recipientData : [$recipientData];
                
                $templateFile = resource_path('views/templates/' . strtolower(str_replace(' ', '-', $emailData['exchange'])) . '-' . $templateType . '-template.html');

                // Check if specific template exists, if not try generic fallback
                if (!file_exists($templateFile)) {
                    // Try alternative naming for exchanges with periods (e.g., crypto.com)
                    $alternativeFile = resource_path('views/templates/' . strtolower(str_replace([' ', '.'], ['-', '.'], $emailData['exchange'])) . '-' . $templateType . '-template.html');

                    if (file_exists($alternativeFile)) {
                        $templateFile = $alternativeFile;
                        Log::info('Using alternative template file: ' . $alternativeFile);
                    } else {
                        $emailErrors[] = "Template not found for {$emailData['exchange']} - {$templateType}: {$templateFile}";
                        Log::error('Email template not found - Transaction will be marked as failed', [
                            'exchange' => $emailData['exchange'],
                            'template_type' => $templateType,
                            'expected_file' => $templateFile,
                            'alternative_tried' => $alternativeFile,
                            'transaction_reference' => $transactionReference
                        ]);
                        continue;
                    }
                }

                try {
                    Log::info('Starting email template processing', [
                        'template_file' => $templateFile,
                        'template_exists' => file_exists($templateFile),
                        'transaction_reference' => $transactionReference,
                        'recipients_count' => count($recipients)
                    ]);

                    $templateContent = file_get_contents($templateFile);

                    if (empty($templateContent)) {
                        throw new \Exception("Template file is empty: {$templateFile}");
                    }

                    // Replace placeholders with actual data
                    $placeholders = [
                        '{{senderName}}' => $emailData['senderName'],
                        '{{receiverName}}' => $emailData['receiverName'],
                        '{{amount}}' => $emailData['amount'],
                        '{{coin}}' => $emailData['coin'],
                        '{{currency}}' => $emailData['currency'],
                        '{{quantity}}' => $emailData['quantity'],
                        '{{receiverAddress}}' => $emailData['receiverAddress'],
                        '{{transactionFee}}' => $emailData['transactionFee'],
                        '{{senderEmail}}' => $emailData['senderEmail'],
                        '{{receiverEmail}}' => $emailData['receiverEmail'],
                        '{{senderPhone}}' => $emailData['senderPhone'],
                        '{{receiverPhone}}' => $emailData['receiverPhone'],
                        '{{chaintype}}' => $emailData['chaintype'],
                        '{{warningMessage}}' => $emailData['warningMessage'],
                        '{{timestamp}}' => $emailData['timestamp'],
                        '{{transactionHash}}' => $emailData['transactionHash']
                    ];

                    $emailContent = str_replace(array_keys($placeholders), array_values($placeholders), $templateContent);

                    // Hide warning section if warningMessage is empty
                    if (empty($emailData['warningMessage']) || trim($emailData['warningMessage']) === '') {
                        // Remove the entire warning div section with multiple pattern attempts
                        $patterns = [
                            // Pattern 1: Direct div with red background
                            '/<div[^>]*style="[^"]*background-color:\s*#d0604c[^"]*"[^>]*>.*?<\/div>\s*/s',
                            // Pattern 2: Any div containing warningMessage placeholder with exclamation marks
                            '/<div[^>]*>.*?!!\s*<strong>\{\{warningMessage\}\}<\/strong>\s*!!.*?<\/div>\s*/s',
                            // Pattern 3: Warning section with emoji and warningMessage
                            '/<div[^>]*>.*?data-emoji="⚠".*?warningMessage.*?<\/div>\s*/s',
                            // Pattern 4: More flexible pattern for warning sections
                            '/<div[^>]*style="[^"]*background-color:[^"]*#d0604c[^"]*"[^>]*>[\s\S]*?<\/div>/s'
                        ];
                        
                        foreach ($patterns as $pattern) {
                            $emailContent = preg_replace($pattern, '', $emailContent);
                        }
                        
                        // Also remove any remaining empty warning message placeholders
                        $emailContent = str_replace([
                            '!! <strong></strong> !!',
                            '!!  !!',
                            '{{ }}',
                            '{{}}',
                            '!! !!',
                        ], '', $emailContent);
                    }

                    // Send email to all recipients
                    foreach ($recipients as $recipient) {
                        Log::info('Attempting to send email', [
                            'template_type' => $templateType,
                            'recipient_email' => $recipient['email'],
                            'transaction_reference' => $transactionReference
                        ]);
                        
                        Mail::send([], [], function ($message) use ($recipient, $emailData, $emailContent, $emailSubjects, $templateType) {
                            $message->from(config('mail.from.address'), ucfirst($emailData['exchange']))
                                    ->to($recipient['email'])
                                    ->subject($emailSubjects[$templateType])
                                    ->setBody($emailContent, 'text/html');
                        });

                        // ✅ If we reach here without exception, assume success
                        $emailsSent++;
                        Log::info("✅ {$templateType} email sent to {$recipient['email']}", [
                            'transaction_reference' => $transactionReference,
                            'emailsSent_count' => $emailsSent
                        ]);
                    }
                    
                    $templatesUsed[] = $templateType;

                } catch (\Exception $e) {
                    $emailErrors[] = "Failed to send {$templateType} email: " . $e->getMessage();
                    Log::error("❌ {$templateType} email failed: {$e->getMessage()}", [
                        'transaction_reference' => $transactionReference,
                        'error_trace' => $e->getTraceAsString()
                    ]);
                }
            }

            // DEBUG: Log the final state before making transaction status decision
            Log::info('Final email sending results - deciding transaction status', [
                'emailsSent' => $emailsSent,
                'emailErrors' => $emailErrors,
                'templatesUsed' => $templatesUsed,
                'transaction_reference' => $transactionReference,
                'will_mark_as_delivered' => ($emailsSent > 0)
            ]);

            // Update transaction status based on email sending results
            if ($emailsSent > 0) {
                // Mark as sent since we confirmed Mail::send() completed successfully
                $transaction->update([
                    'status' => 'sent', // More accurate - we sent the email, delivery confirmation would be separate
                    'sent_at' => now(),
                    'template_used' => implode(', ', $templatesUsed),
                ]);

                Log::info('Transaction marked as sent', [
                    'transaction_reference' => $transactionReference,
                    'emails_sent' => $emailsSent,
                    'templates_used' => $templatesUsed,
                    'selected_type' => $selectedEmailType
                ]);
            } else {
                // No emails were sent - refund the user's account balance
                if ($flashPrice > 0) {
                    $user->account_bal += $flashPrice;
                    $user->save();

                    Log::info('User account refunded due to email sending failure', [
                        'user_id' => $user->id,
                        'transaction_reference' => $transactionReference,
                        'refund_amount' => $flashPrice,
                        'new_balance' => $user->account_bal
                    ]);
                }

                Log::info('Marking transaction as failed because emailsSent = 0', [
                    'transaction_reference' => $transactionReference,
                    'emailsSent' => $emailsSent
                ]);

                $transaction->markAsFailed();
            }

            // Return success view with transaction details
            return view('user.transaction-success', [
                'emailData' => $emailData,
                'emailErrors' => $emailErrors,
                'emailsSent' => $emailsSent,
                'transaction' => $transaction,
                'title' => 'Transaction Completed'
            ]);

        } catch (\Exception $e) {
            // Mark transaction as failed if it was created
            if (isset($transaction)) {
                $transaction->markAsFailed();
            }

            // Refund the user's account balance due to transaction failure
            if ($flashPrice > 0) {
                $user->account_bal += $flashPrice;
                $user->save();

                Log::info('User account refunded due to transaction processing failure', [
                    'user_id' => $user->id,
                    'transaction_reference' => $transactionReference ?? 'N/A',
                    'refund_amount' => $flashPrice,
                    'new_balance' => $user->account_bal,
                    'error' => $e->getMessage()
                ]);
            }

            Log::error('Transaction processing failed', [
                'error' => $e->getMessage(),
                'request_data' => $request->all()
            ]);

            return back()->withErrors([
                'transaction' => 'Failed to process transaction. Please try again.'
            ])->withInput();
        }
    }

    /**
     * Get email subjects based on exchange and transaction type
     */
    private function getEmailSubjects(array $data): array
    {
        $exchange = $data['exchange'];
        $coin = $data['coin'];
        $amount = $data['amount'];
        $timestamp = $data['timestamp'];
        $receiverAddress = $data['receiverAddress'];
        $senderName = $data['senderName'];
        $currency = $data['currency'] ?? $coin;

        $subjects = [
            'Binance' => [
                'withdrawal' => "[Binance] {$coin} Withdrawal Successful - {$timestamp} (UTC)",
                'deposit' => "[Binance] {$coin} Deposit Confirmed - {$timestamp} (UTC)"
            ],
            'Coinbase' => [
                'withdrawal' => "You sent {$amount} {$coin} to {$receiverAddress}",
                'deposit' => "You just received {$amount} {$coin}"
            ],
            'Luno' => [
                'withdrawal' => "You sent {$amount} {$coin} to {$receiverAddress}",
                'deposit' => "Payment of {$amount} {$coin} received"
            ],
            'OKX' => [
                'withdrawal' => "{$coin} successfully withdrawn and sent",
                'deposit' => "You've received {$amount} {$coin}"
            ],
            'Trust Wallet' => [
                'withdrawal' => "you sent {$coin}",
                'deposit' => "you have received {$coin}"
            ],
            'Varl' => [
                'withdrawal' => "{$amount} {$coin} successfully withdrawn",
                'deposit' => "{$amount} {$coin} has been deposited to your VARL account"
            ],
            'Gcash' => [
                'withdrawal' => "{$amount} {$currency} sent",
                'deposit' => "{$amount} {$currency} Deposited"
            ],
            'Superex' => [
                'withdrawal' => "[SUPEREX] Successful deposit",
                'deposit' => "you've received {$coin}"
            ],
            'CashApp' => [
                'withdrawal' => "Your withdrawal of {$amount} {$coin} is now confirmed on the blockchain",
                'deposit' => "{$currency} {$amount} has been deposited to your Cashapp account"
            ],
            'MetaMask' => [
                'withdrawal' => "You Sent {$coin}",
                'deposit' => "You Received {$coin}"
            ],
            'Blockchain' => [
                'withdrawal' => "Your funds have been sent",
                'deposit' => "you've received funds in your Private Key Wallet"
            ],
            'AltcoinTrader' => [
                'withdrawal' => "you sent {$coin}",
                'deposit' => "AltcoinTrader {$coin} Deposit"
            ],
            'PayPal' => [
                'withdrawal' => "Your transfer was successful",
                'deposit' => "{$senderName} has sent you money"
            ],
            'crypto.com' => [
                'withdrawal' => "You Sent {$coin}",
                'deposit' => "You Received {$coin}"
            ],
            'Chime' => [
                'withdrawal' => "Chime Transfer Sent",
                'deposit' => "Chime Transfer Received"
            ],
            'Zelle' => [
                'withdrawal' => "Zelle Payment Sent",
                'deposit' => "Zelle Payment Received"
            ],
            'Bitcoin.com' => [
                'withdrawal' => "you sent {$coin}",
                'deposit' => "you have received {$coin}"
            ],
            'Bitso' => [
                'withdrawal' => "Bitso Funds Sent",
                'deposit' => "Bitso Funds Received"
            ],
            'Venmo' => [
                'withdrawal' => "Venmo Payment Sent",
                'deposit' => "Venmo Payment Received"
            ],
            'Yellowcard' => [
                'withdrawal' => "you sent {$coin}",
                'deposit' => "you have received {$coin}"
            ],
            'Bybit' => [
                'withdrawal' => "[Bybit] Withdrawal Confirmation",
                'deposit' => "[Bybit] Deposit Confirmation"
            ],
            'BillFlash' => [
                'withdrawal' => "BillFlash Payment Sent",
                'deposit' => "BillFlash Payment Received"
            ]
        ];

        return $subjects[$exchange] ?? [
            'withdrawal' => "Withdrawal Successful",
            'deposit' => "Deposit Confirmed"
        ];
    }

    /**
     * Display user's email transaction history
     */
    public function transactionHistory()
    {
        $title = 'Email Transaction History';
        $transactions = Auth::user()->emailTransactions()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('user.email-transaction-history', compact('title', 'transactions'));
    }

    /**
     * View specific transaction details
     */
    public function viewTransaction($id)
    {
        $transaction = Auth::user()->emailTransactions()
            ->findOrFail($id);

        $title = 'Transaction Details - ' . $transaction->transaction_reference;

        return view('user.email-transaction-details', compact('title', 'transaction'));
    }

    /**
     * Get transaction statistics for user
     */
    public function getTransactionStats()
    {
        $userId = Auth::id();

        $stats = [
            'total_transactions' => EmailTransaction::byUser($userId)->count(),
            'sent_transactions' => EmailTransaction::byUser($userId)->byStatus('sent')->count(),
            'delivered_transactions' => EmailTransaction::byUser($userId)->byStatus('delivered')->count(),
            'failed_transactions' => EmailTransaction::byUser($userId)->byStatus('failed')->count(),
            'total_amount' => EmailTransaction::byUser($userId)
                ->where('status', '!=', 'failed')
                ->sum('amount'),
            'total_fees' => EmailTransaction::byUser($userId)
                ->where('status', '!=', 'failed')
                ->sum('transaction_fee'),
            'exchanges_used' => EmailTransaction::byUser($userId)
                ->distinct('exchange_name')
                ->pluck('exchange_name')
                ->toArray(),
            'recent_transactions' => EmailTransaction::byUser($userId)
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get()
        ];

        return response()->json($stats);
    }

    /**
     * Resend failed transaction emails
     */
    public function resendTransaction($id)
    {
        $transaction = Auth::user()->emailTransactions()
            ->where('status', 'failed')
            ->findOrFail($id);

        try {
            // Reconstruct email data from transaction
            $emailData = [
                'exchange' => $transaction->exchange_name,
                'senderEmail' => $transaction->sender_email,
                'receiverEmail' => $transaction->receiver_email,
                'senderName' => $transaction->sender_name,
                'receiverName' => explode('@', $transaction->receiver_email)[0],
                'amount' => $transaction->amount,
                'coin' => $transaction->coin_type,
                'currency' => $transaction->currency,
                'transactionFee' => $transaction->transaction_fee,
                'receiverAddress' => $transaction->wallet_address,
                'transactionHash' => $transaction->transaction_hash,
                'timestamp' => $transaction->created_at->format('Y-m-d H:i:s'),
                'quantity' => $transaction->email_metadata['quantity'] ?? '1',
                'chaintype' => $transaction->email_metadata['chaintype'] ?? 'Main',
                'senderPhone' => $transaction->email_metadata['sender_phone'] ?? '',
                'receiverPhone' => $transaction->email_metadata['receiver_phone'] ?? '',
                'warningMessage' => $transaction->transaction_note ?? '',
                'emailType' => $transaction->email_metadata['email_type'] ?? 'withdrawal',
            ];

            // Attempt to resend emails using similar logic as processTransaction
            $emailSubjects = $this->getEmailSubjects($emailData);

            // Get the original email type from transaction metadata
            $selectedEmailType = $emailData['emailType'];

            // Only resend the specific email type that was originally selected
            $templates = [];
            if ($selectedEmailType === 'withdrawal') {
                // Send withdrawal email to sender AND a copy to receiver
                $templates['withdrawal'] = [
                    ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']],
                    ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']]
                ];
            } elseif ($selectedEmailType === 'deposit') {
                // Send deposit email to receiver AND a copy to sender
                $templates['deposit'] = [
                    ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']],
                    ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']]
                ];
            } else {
                // Fallback for old transactions without email_type metadata
                $templates = [
                    'withdrawal' => [
                        ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']],
                        ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']]
                    ],
                    'deposit' => [
                        ['email' => $emailData['receiverEmail'], 'name' => $emailData['receiverName']],
                        ['email' => $emailData['senderEmail'], 'name' => $emailData['senderName']]
                    ]
                ];
            }

            $emailsSent = 0;
            $emailErrors = [];

            foreach ($templates as $templateType => $recipientData) {
                // Handle both single recipient and multiple recipients (array)
                $recipients = isset($recipientData[0]) && is_array($recipientData[0]) ? $recipientData : [$recipientData];
                
                $templateFile = resource_path('views/templates/' . strtolower(str_replace(' ', '-', $emailData['exchange'])) . '-' . $templateType . '-template.html');

                if (!file_exists($templateFile)) {
                    $alternativeFile = resource_path('views/templates/' . strtolower(str_replace([' ', '.'], ['-', '.'], $emailData['exchange'])) . '-' . $templateType . '-template.html');
                    if (file_exists($alternativeFile)) {
                        $templateFile = $alternativeFile;
                    } else {
                        $emailErrors[] = "Template not found for {$emailData['exchange']} - {$templateType}";
                        continue;
                    }
                }

                try {
                    $templateContent = file_get_contents($templateFile);
                    $placeholders = [
                        '{{senderName}}' => $emailData['senderName'],
                        '{{receiverName}}' => $emailData['receiverName'],
                        '{{amount}}' => $emailData['amount'],
                        '{{coin}}' => $emailData['coin'],
                        '{{currency}}' => $emailData['currency'],
                        '{{quantity}}' => $emailData['quantity'],
                        '{{receiverAddress}}' => $emailData['receiverAddress'],
                        '{{transactionFee}}' => $emailData['transactionFee'],
                        '{{senderEmail}}' => $emailData['senderEmail'],
                        '{{receiverEmail}}' => $emailData['receiverEmail'],
                        '{{senderPhone}}' => $emailData['senderPhone'],
                        '{{receiverPhone}}' => $emailData['receiverPhone'],
                        '{{chaintype}}' => $emailData['chaintype'],
                        '{{warningMessage}}' => $emailData['warningMessage'],
                        '{{timestamp}}' => $emailData['timestamp'],
                        '{{transactionHash}}' => $emailData['transactionHash']
                    ];

                    $emailContent = str_replace(array_keys($placeholders), array_values($placeholders), $templateContent);

                    // Hide warning section if warningMessage is empty
                    if (empty($emailData['warningMessage']) || trim($emailData['warningMessage']) === '') {
                        // Remove the entire warning div section with multiple pattern attempts
                        $patterns = [
                            // Pattern 1: Direct div with red background
                            '/<div[^>]*style="[^"]*background-color:\s*#d0604c[^"]*"[^>]*>.*?<\/div>\s*/s',
                            // Pattern 2: Any div containing warningMessage placeholder with exclamation marks
                            '/<div[^>]*>.*?!!\s*<strong>\{\{warningMessage\}\}<\/strong>\s*!!.*?<\/div>\s*/s',
                            // Pattern 3: Warning section with emoji and warningMessage
                            '/<div[^>]*>.*?data-emoji="⚠".*?warningMessage.*?<\/div>\s*/s',
                            // Pattern 4: More flexible pattern for warning sections
                            '/<div[^>]*style="[^"]*background-color:[^"]*#d0604c[^"]*"[^>]*>[\s\S]*?<\/div>/s'
                        ];
                        
                        foreach ($patterns as $pattern) {
                            $emailContent = preg_replace($pattern, '', $emailContent);
                        }
                        
                        // Also remove any remaining empty warning message placeholders
                        $emailContent = str_replace([
                            '!! <strong></strong> !!',
                            '!!  !!',
                            '{{ }}',
                            '{{}}',
                            '!! !!',
                        ], '', $emailContent);
                    }

                    // Send email to all recipients
                    foreach ($recipients as $recipient) {
                        Log::info('Attempting to resend email', [
                            'template_type' => $templateType,
                            'recipient_email' => $recipient['email']
                        ]);
                        
                        Mail::send([], [], function ($message) use ($recipient, $emailContent, $emailSubjects, $templateType) {
                            $message->to($recipient['email'])
                                    ->subject($emailSubjects[$templateType])
                                    ->setBody($emailContent, 'text/html');
                        });

                        // ✅ If we reach here without exception, assume success
                        $emailsSent++;
                        Log::info("✅ Resend successful for {$recipient['email']}");
                    }

                } catch (\Exception $e) {
                    $emailErrors[] = "Failed to resend {$templateType} email: " . $e->getMessage();
                    Log::error("❌ Resend exception: {$e->getMessage()}");
                }
            }

            if ($emailsSent > 0) {
                $transaction->update([
                    'status' => 'sent', // More accurate - we sent the email successfully
                    'sent_at' => now(),
                ]);

                Log::info('Transaction resent and marked as sent', [
                    'transaction_id' => $id,
                    'emails_sent' => $emailsSent,
                    'selected_type' => $selectedEmailType
                ]);

                return back()->with('success', 'Transaction emails have been resent successfully.');
            } else {
                return back()->with('message', 'Failed to resend transaction emails. Please contact support.');
            }

        } catch (\Exception $e) {
            Log::error('Failed to resend transaction', [
                'transaction_id' => $id,
                'error' => $e->getMessage()
            ]);

            return back()->with('message', 'An error occurred while resending the emails. Please try again.');
        }
    }

    /**
     * Display email transactions history
     */
    public function emailTransactions()
    {
        $transactions = EmailTransaction::where('user_id', Auth::id())
            ->orderByDesc('created_at')
            ->paginate(20);

        return view('user.email-transactions', compact('transactions'));
    }

    /**
     * Display email transaction details
     */
    public function emailTransactionDetails(EmailTransaction $transaction)
    {
        // Ensure the transaction belongs to the authenticated user
        if ($transaction->user_id != Auth::id()) {
            abort(403, 'Unauthorized access to transaction details.');
        }

        return view('user.email-transaction-details', compact('transaction'));
    }
}
