<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Settings;
use App\Models\Agent;
use App\Models\Deposit;
use App\Models\Withdrawal;
use App\Models\Activity;
use App\Models\EmailTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Mail\NewNotification;
use App\Models\Kyc;
use App\Models\Mt4Details;
use App\Models\Booking;
use App\Models\Flight;
use App\Traits\PingServer;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Mail;

class ManageUsersController extends Controller
{
    use PingServer;

    // See user wallet balances
    public function loginactivity($id)
    {

        $user = User::where('id', $id)->first();

        return view('admin.Users.loginactivity', [
            'activities' => Activity::where('user', $id)->orderByDesc('id')->get(),
            'title' => "$user->name login activities",
            'user' => $user,
        ]);
    }

    public function showUsers($id)
    {
        $user = User::where('id', $id)->first();
        $ref = User::whereNull('ref_by')->where('id', '!=', $id)->get();

        return view('admin.Users.referral', [
            'title' => "Add users to $user->name referral list",
            'user' => $user,
            'ref' => $ref,
        ]);
    }

    public function fetchUsers()
    {
        $users = User::orderByDesc('id')->get();
        return response()->json([
            'message' => 'Success',
            'data' => $users,
            'code' => 200
        ]);
    }


    public function addReferral(Request $request)
    {
        $user = User::where('id', $request->user_id)->first();
        $ref = User::where('id', $request->ref_id)->first();

        $ref->ref_by = $user->id;
        $ref->save();
        return redirect()->back()
            ->with('success', "$ref->name is now referred by $user->name successfully");
    }

    public function clearactivity($id)
    {
        $activities = Activity::where('user', $id)->get();

        if (count($activities) > 0) {
            foreach ($activities as $act) {
                Activity::where('id', $act->id)->delete();
            }
            return redirect()->back()
                ->with('success', 'Activity Cleared Successfully!');
        }
        return redirect()->back()
            ->with('message', 'No Activity to clear!');
    }

    public function viewuser($id)
    {
        $user = User::where('id', $id)->first();

        // Get user's flight bookings
        $bookings = Booking::with(['flight.originAirport', 'flight.destinationAirport'])
            ->where('user_id', $id)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Flight booking stats for this user
        $stats = [
            'total' => Booking::where('user_id', $id)->count(),
            'confirmed' => Booking::where('user_id', $id)->where('status', 'confirmed')->count(),
            'pending' => Booking::where('user_id', $id)->where('payment_status', 'pending')->count(),
            'cancelled' => Booking::where('user_id', $id)->where('status', 'cancelled')->count(),
            'completed' => Booking::where('user_id', $id)->where('status', 'completed')->count(),
            'total_spent' => Booking::where('user_id', $id)->where('payment_status', 'paid')->sum('price_paid')
        ];

        return view('admin.Users.userdetails', [
            'user' => $user,
            'bookings' => $bookings,
            'stats' => $stats,
            'title' => "Manage $user->name",
        ]);
    }
    //block user
    public function ublock($id)
    {
        User::where('id', $id)->update([
            'status' => 'blocked',
        ]);
        return redirect()->back()->with('success', 'Action Sucessful!');
    }

    //unblock user
    public function unblock($id)
    {
        User::where('id', $id)->update([
            'status' => 'active',
        ]);
        return redirect()->back()->with('success', 'Action Sucessful!');
    }

    //Turn on/off user trade
    public function usertrademode($id, $action)
    {
        if ($action == "on") {
            $action = "on";
        } elseif ($action == "off") {
            $action = "off";
        } else {
            return redirect() - back()->with('message', "Unknown action!");
        }

        User::where('id', $id)->update([
            'trade_mode' => $action,
        ]);
        return redirect()->back()->with('success', "User trade mode has been turned $action.");
    }

    //Manually Verify users email
    public function emailverify($id)
    {
        User::where('id', $id)->update([
            'email_verified_at' => \Carbon\Carbon::now(),
        ]);
        return redirect()->back()->with('success', 'User Email have been verified');
    }

    //Reset Password
    public function resetpswd($id)
    {
        User::where('id', $id)
            ->update([
                'password' => Hash::make('user01236'),
            ]);
        return redirect()->back()->with('success', 'Password has been reset to default');
    }

    //Clear user Account
    public function clearacct(Request $request, $id)
    {
        $settings = Settings::where('id', 1)->first();

        $deposits = Deposit::where('user', $id)->get();
        if (!empty($deposits)) {
            foreach ($deposits as $deposit) {
                Deposit::where('id', $deposit->id)->delete();
            }
        }

        $withdrawals = Withdrawal::where('user', $id)->get();
        if (!empty($withdrawals)) {
            foreach ($withdrawals as $withdrawals) {
                Withdrawal::where('id', $withdrawals->id)->delete();
            }
        }

        User::where('id', $id)->update([
            'account_bal' => '0',
            'roi' => '0',
            'bonus' => '0',
            'ref_bonus' => '0',
        ]);
        return redirect()->back()->with('success', "Account cleared to $settings->currency 0.00");
    }

    //Access users account
    public function switchuser($id)
    {
        $user = User::where('id', $id)->first();
        Auth::loginUsingId($user->id, true);
        return redirect()->route('dashboard')->with('success', "You are logged in as $user->name !");
    }

    //Delete user
    public function delsystemuser($id)
    {
        try {
            DB::beginTransaction();

            // Delete the user's flight bookings first (due to foreign key constraint)
            $bookings = Booking::where('user_id', $id)->get();
            foreach ($bookings as $booking) {
                // Return seats to flight availability before deleting
                if ($booking->flight) {
                    $booking->flight->increment('seats_available');
                }
                $booking->delete();
            }

            // Handle flights created by this admin user - set created_by to null to preserve flights
            Flight::where('created_by', $id)->update(['created_by' => null]);

            
            // Delete the user's activities
            Activity::where('user', $id)->delete();

            // Finally delete the user
            User::where('id', $id)->delete();

            DB::commit();

            return redirect()->route('manageusers')
                ->with('success', 'User Account deleted successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('message', 'Error deleting user: ' . $e->getMessage());
        }
    }

    //update users info
    public function edituser(Request $request)
    {

        User::where('id', $request['user_id'])
            ->update([
                'passenger_name' => $request['name'],
                'passenger_email' => $request['email'],
                'country' => $request['country'],
                'username' => $request['username'],
                'passenger_phone' => $request['phone'],
            ]);
        return redirect()->back()->with('success', 'User details updated Successfully!');
    }

    //Send mail to one user
    public function sendmailtooneuser(Request $request)
    {

        $mailduser = User::where('id', $request->user_id)->first();
        Mail::to($mailduser->email)->send(new NewNotification($request->message, $request->subject, $mailduser->name));
        return redirect()->back()->with('success', 'Your message was sent successfully!');
    }

    // Send Mail to all users
    public function sendmailtoall(Request $request)
    {

        if ($request->category == "All") {
            User::select(['email', 'id'])->chunkById(100, function ($users) use ($request) {
                foreach ($users as $user) {
                    Mail::to($user->email)->send(new NewNotification($request->message, $request->subject, $request->title, null, null, $request->greet));
                }
            });
        } elseif ($request->category == "No deposit") {
            User::doesntHave('dp')->select(['email', 'id'])->chunkById(100, function ($users) use ($request) {
                foreach ($users as $user) {
                    Mail::to($user->email)->send(new NewNotification($request->message, $request->subject, $request->title, null, null, $request->greet));
                }
            });
        } elseif ($request->category == "Select Users") {
            DB::table('users')
                ->whereIn('id', array_column($request->users, null))
                ->select(['email', 'id'])
                ->chunkById(100, function ($users) use ($request) {
                    foreach ($users as $user) {
                        Mail::to($user->email)->send(new NewNotification($request->message, $request->subject, $request->title, null, null, $request->greet));
                    }
                });
        }

        return redirect()->back()->with('success', 'Your message was sent successfully!');
    }

    // Save new user
    public function saveuser(Request $request)
    {
        $request->validate([
            'name' => 'required|max:255',
            'username' => 'required|unique:users,username',
            'email' => 'required|email|max:255|unique:users,email|unique:users,passenger_email',
            'phone' => 'required',
            'password' => 'required|min:8',
        ]);

        $user = User::create([
            'name' => $request->name,
            'passenger_name' => $request->name,
            'email' => $request->email,
            'passenger_email' => $request->email,
            'passenger_phone' => $request->phone,
            'passenger_dob' => $request->passenger_dob,
            'nationality' => $request->nationality,
            'passport_number' => $request->passport_number,
            'passport_country' => $request->passport_country,
            'passport_expiry' => $request->passport_expiry,
            'username' => $request->username,
            'password' => Hash::make($request->password),
            'status' => 'active',
        ]);

        // Assign referral link to user
        $settings = Settings::first();
        if ($user && $settings) {
            $user->update([
                'ref_link' => $settings->site_address . '/ref/' . $user->username,
            ]);
        }
        
        return redirect()->back()->with('success', 'User created successfully!');
    }
}
