<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Flight;
use App\Models\Airport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class FlightController extends Controller
{
    public function __construct()
    {
        $this->middleware('isadmin');
    }

    /**
     * Display a listing of flights.
     */
    public function index(Request $request)
    {
        $query = Flight::with(['originAirport', 'destinationAirport'])
            ->orderBy('created_at', 'desc');

        // Filter by status if provided
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('departure_time', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('departure_time', '<=', $request->date_to);
        }

        // Search by flight number or airline
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('flight_number', 'like', "%{$search}%")
                  ->orWhere('airline', 'like', "%{$search}%");
            });
        }

        $flights = $query->paginate(20)->withQueryString();

        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Flight Management';

        return view('admin.flights.index', compact('flights', 'gradient', 'title'));
    }

    /**
     * Show the form for creating a new flight.
     */
    public function create()
    {
        $airports = Airport::active()->orderBy('iata_code')->get();
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Add New Flight';
        
        return view('admin.flights.create', compact('airports', 'gradient', 'title'));
    }

    /**
     * Store a newly created flight.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'airline' => 'required|string|max:255',
            'airline_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'flight_number' => 'required|string|max:10',
            'origin_airport_id' => 'required|exists:airports,id',
            'destination_airport_id' => 'required|exists:airports,id|different:origin_airport_id',
            'departure_time' => 'required|date_format:H:i',
            'arrival_time' => 'required|date_format:H:i',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'required|date|after_or_equal:start_date',
            'frequency' => 'required|in:daily,every_other_day,weekdays,weekends,weekly',
            'terminal' => 'nullable|string|max:10',
            'gate' => 'nullable|string|max:10',
            'aircraft_type' => 'nullable|string|max:50',
            'seats_total' => 'required|integer|min:1|max:1000',
            'price' => 'required|numeric|min:0',
        ]);

        // Handle airline logo upload
        $airlineLogo = null;
        if ($request->hasFile('airline_logo')) {
            $airlineLogo = $request->file('airline_logo')->store('airline-logos', 'public');
        }

        // Calculate flight duration in minutes
        $departureTime = \Carbon\Carbon::createFromFormat('H:i', $validated['departure_time']);
        $arrivalTime = \Carbon\Carbon::createFromFormat('H:i', $validated['arrival_time']);
        
        // Handle overnight flights (arrival next day)
        if ($arrivalTime->lessThan($departureTime)) {
            $arrivalTime->addDay();
        }
        
        $flightDurationMinutes = $departureTime->diffInMinutes($arrivalTime);

        // Generate flight instances based on date range and frequency
        $startDate = \Carbon\Carbon::parse($validated['start_date']);
        $endDate = \Carbon\Carbon::parse($validated['end_date']);
        $currentDate = $startDate->copy();
        $flightsCreated = 0;
        $flightData = [];

        while ($currentDate->lessThanOrEqualTo($endDate)) {
            $shouldCreate = false;

            switch ($validated['frequency']) {
                case 'daily':
                    $shouldCreate = true;
                    break;
                case 'every_other_day':
                    $shouldCreate = $currentDate->diffInDays($startDate) % 2 == 0;
                    break;
                case 'weekdays':
                    $shouldCreate = $currentDate->isWeekday();
                    break;
                case 'weekends':
                    $shouldCreate = $currentDate->isWeekend();
                    break;
                case 'weekly':
                    $shouldCreate = $currentDate->diffInDays($startDate) % 7 == 0;
                    break;
            }

            if ($shouldCreate) {
                // Create departure datetime
                $departure = $currentDate->copy()
                    ->setTimeFromTimeString($validated['departure_time']);
                
                // Create arrival datetime (may be next day for overnight flights)
                $arrival = $currentDate->copy()
                    ->setTimeFromTimeString($validated['arrival_time']);
                
                if ($arrival->lessThan($departure)) {
                    $arrival->addDay();
                }

                // Check if flight already exists
                $exists = Flight::where('airline', $validated['airline'])
                    ->where('flight_number', $validated['flight_number'])
                    ->whereDate('departure_time', $departure->format('Y-m-d'))
                    ->exists();

                if (!$exists) {
                    $flightData[] = [
                        'airline' => $validated['airline'],
                        'airline_logo' => $airlineLogo,
                        'flight_number' => $validated['flight_number'],
                        'origin_airport_id' => $validated['origin_airport_id'],
                        'destination_airport_id' => $validated['destination_airport_id'],
                        'departure_time' => $departure,
                        'arrival_time' => $arrival,
                        'terminal' => $validated['terminal'] ?? null,
                        'gate' => $validated['gate'] ?? null,
                        'aircraft_type' => $validated['aircraft_type'] ?? null,
                        'seats_total' => $validated['seats_total'],
                        'seats_available' => $validated['seats_total'],
                        'price' => $validated['price'],
                        'status' => 'scheduled',
                        'created_by' => auth('admin')->id(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                    $flightsCreated++;
                }
            }

            $currentDate->addDay();
        }

        // Bulk insert flights
        if (count($flightData) > 0) {
            Flight::insert($flightData);
        }

        $message = $flightsCreated > 0 
            ? "Successfully created {$flightsCreated} flight instance(s) from {$startDate->format('M d, Y')} to {$endDate->format('M d, Y')}!"
            : 'No new flights were created. All flights for the selected dates already exist.';

        return redirect()->route('admin.flights.index')
            ->with('success', $message);
    }

    /**
     * Display the specified flight.
     */
    public function show(Flight $flight)
    {
        $flight->load(['originAirport', 'destinationAirport', 'bookings.user']);
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Flight Details';
        
        return view('admin.flights.show', compact('flight', 'gradient', 'title'));
    }

    /**
     * Show the form for editing the specified flight.
     */
    public function edit(Flight $flight)
    {
        $airports = Airport::active()->orderBy('iata_code')->get();
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Edit Flight';
        
        return view('admin.flights.edit', compact('flight', 'airports', 'gradient', 'title'));
    }

    /**
     * Update the specified flight.
     */
    public function update(Request $request, Flight $flight)
    {
        $validated = $request->validate([
            'airline' => 'required|string|max:255',
            'airline_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'flight_number' => 'required|string|max:10',
            'origin_airport_id' => 'required|exists:airports,id',
            'destination_airport_id' => 'required|exists:airports,id|different:origin_airport_id',
            'departure_time' => 'required|date',
            'arrival_time' => 'required|date|after:departure_time',
            'status' => 'required|in:scheduled,boarding,departed,airborne,delayed,landed,cancelled',
            'terminal' => 'nullable|string|max:10',
            'gate' => 'nullable|string|max:10',
            'aircraft_type' => 'nullable|string|max:50',
            'seats_total' => 'required|integer|min:1|max:1000',
            'price' => 'required|numeric|min:0',
            'tracking_lat' => 'nullable|numeric|between:-90,90',
            'tracking_lng' => 'nullable|numeric|between:-180,180',
            'altitude' => 'nullable|integer|min:0|max:60000',
            'speed' => 'nullable|integer|min:0|max:1000',
        ]);

        // Handle airline logo upload
        if ($request->hasFile('airline_logo')) {
            // Delete old logo if exists
            if ($flight->airline_logo) {
                Storage::disk('public')->delete($flight->airline_logo);
            }
            $validated['airline_logo'] = $request->file('airline_logo')->store('airline-logos', 'public');
        }

        // Update available seats if total seats changed
        if ($validated['seats_total'] != $flight->seats_total) {
            $bookedSeats = $flight->seats_total - $flight->seats_available;
            $validated['seats_available'] = max(0, $validated['seats_total'] - $bookedSeats);
        }

        // Update position timestamp if position changed
        if (($validated['tracking_lat'] ?? null) !== $flight->tracking_lat || 
            ($validated['tracking_lng'] ?? null) !== $flight->tracking_lng) {
            $validated['last_position_updated_at'] = now();
        }

        $flight->update($validated);

        return redirect()->route('admin.flights.index')
            ->with('success', 'Flight updated successfully!');
    }

    /**
     * Update flight tracking position via AJAX.
     */
    public function updatePosition(Request $request, Flight $flight)
    {
        $validated = $request->validate([
            'lat' => 'required|numeric|between:-90,90',
            'lng' => 'required|numeric|between:-180,180',
            'altitude' => 'nullable|integer|min:0|max:60000',
            'speed' => 'nullable|integer|min:0|max:1000',
        ]);

        $flight->updatePosition(
            $validated['lat'],
            $validated['lng'],
            $validated['altitude'] ?? null,
            $validated['speed'] ?? null
        );

        return response()->json([
            'success' => true,
            'message' => 'Flight position updated successfully',
            'data' => [
                'lat' => $flight->tracking_lat,
                'lng' => $flight->tracking_lng,
                'altitude' => $flight->altitude,
                'speed' => $flight->speed,
                'updated_at' => $flight->last_position_updated_at,
            ]
        ]);
    }

    /**
     * Update flight status via AJAX.
     */
    public function updateStatus(Request $request, Flight $flight)
    {
        $validated = $request->validate([
            'status' => 'required|in:scheduled,boarding,departed,airborne,delayed,landed,cancelled',
            'terminal' => 'nullable|string|max:10',
            'gate' => 'nullable|string|max:10',
        ]);

        $flight->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Flight status updated successfully',
            'data' => [
                'status' => $flight->status,
                'terminal' => $flight->terminal,
                'gate' => $flight->gate,
            ]
        ]);
    }

    /**
     * Remove the specified flight.
     */
    public function destroy(Flight $flight)
    {
        // Check if flight has bookings
        if ($flight->bookings()->where('status', '!=', 'cancelled')->count() > 0) {
            return back()->withErrors(['error' => 'Cannot delete flight with active bookings.']);
        }

        $flight->delete();

        return redirect()->route('admin.flights.index')
            ->with('success', 'Flight deleted successfully!');
    }
}
