<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Flight;
use App\Models\Wdmethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class FlightBookingController extends Controller
{
    /**
     * Display a listing of all bookings
     */
    public function index(Request $request)
    {
        $query = Booking::with(['flight.originAirport', 'flight.destinationAirport', 'user']);
        
        // Filter by payment status
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }
        
        // Filter by booking status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        // Search by booking reference or passenger name
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('booking_reference', 'like', "%{$search}%")
                  ->orWhere('passenger_name', 'like', "%{$search}%")
                  ->orWhere('passenger_email', 'like', "%{$search}%");
            });
        }
        
        $bookings = $query->latest()->paginate(20);
        
        // Get statistics
        $stats = [
            'total' => Booking::count(),
            'pending' => Booking::where('payment_status', 'pending')->count(),
            'paid' => Booking::where('payment_status', 'paid')->count(),
            'today' => Booking::whereDate('created_at', today())->count(),
            'revenue' => Booking::where('payment_status', 'paid')->sum('price_paid'),
        ];
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Flight Bookings Management';
        
        return view('admin.bookings.index', compact('bookings', 'stats', 'gradient', 'title'));
    }

    /**
     * Display pending payments that need approval
     */
    public function pendingPayments()
    {
        $bookings = Booking::with(['flight.originAirport', 'flight.destinationAirport', 'user', 'paymentMethod'])
            ->where('payment_status', 'pending')
            ->latest()
            ->paginate(20);
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Pending Payment Approvals';
            
        return view('admin.bookings.pending-payments', compact('bookings', 'gradient', 'title'));
    }

    /**
     * Display the specified booking
     */
    public function show($id)
    {
        $booking = Booking::with([
            'flight.originAirport',
            'flight.destinationAirport',
            'user'
        ])->findOrFail($id);
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Booking Details';
        
        return view('admin.bookings.show', compact('booking', 'gradient', 'title'));
    }

    /**
     * Approve a payment
     */
    public function approvePayment(Request $request, $id)
    {
        $booking = Booking::findOrFail($id);
        
        if ($booking->payment_status !== 'pending') {
            return redirect()->back()->with('error', 'Payment already processed.');
        }
        
        $booking->update([
            'payment_status' => 'paid',
            'status' => 'confirmed'
        ]);
        
        // Decrease available seats
        $booking->flight->decrement('seats_available', 1);
        
        // TODO: Send confirmation email to passenger
        
        return redirect()->back()->with('success', 'Payment approved successfully. Confirmation email sent to passenger.');
    }

    /**
     * Reject a payment
     */
    public function rejectPayment(Request $request, $id)
    {
        $request->validate([
            'rejection_reason' => 'required|string|max:500'
        ]);
        
        $booking = Booking::findOrFail($id);
        
        if ($booking->payment_status !== 'pending') {
            return redirect()->back()->with('error', 'Payment already processed.');
        }
        
        $booking->update([
            'payment_status' => 'failed',
            'status' => 'cancelled'
        ]);
        
        // TODO: Send rejection email with reason
        
        return redirect()->back()->with('success', 'Payment rejected. Customer has been notified.');
    }

    /**
     * Cancel a booking
     */
    public function cancel($id)
    {
        $booking = Booking::findOrFail($id);
        
        if ($booking->status === 'cancelled') {
            return redirect()->back()->with('error', 'Booking already cancelled.');
        }
        
        $booking->update([
            'status' => 'cancelled',
            'payment_status' => $booking->payment_status === 'paid' ? 'refunded' : 'failed'
        ]);
        
        // Return seat to available inventory if payment was completed
        if ($booking->payment_status === 'refunded') {
            $booking->flight->increment('seats_available', 1);
        }
        
        // TODO: Send cancellation email
        
        return redirect()->back()->with('success', 'Booking cancelled successfully.');
    }

    /**
     * Export bookings to CSV
     */
    public function export(Request $request)
    {
        $bookings = Booking::with(['flight.originAirport', 'flight.destinationAirport', 'user'])
            ->when($request->filled('payment_status'), function($q) use ($request) {
                $q->where('payment_status', $request->payment_status);
            })
            ->when($request->filled('status'), function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->get();
        
        $filename = 'bookings_' . now()->format('Y-m-d_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];
        
        $callback = function() use ($bookings) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, [
                'Booking Reference',
                'Passenger Name',
                'Passenger Email',
                'Flight Number',
                'From',
                'To',
                'Departure',
                'Price',
                'Payment Status',
                'Booking Status',
                'Booked At'
            ]);
            
            // Data rows
            foreach ($bookings as $booking) {
                fputcsv($file, [
                    $booking->booking_reference,
                    $booking->passenger_name,
                    $booking->passenger_email,
                    $booking->flight->flight_number,
                    $booking->flight->originAirport->city,
                    $booking->flight->destinationAirport->city,
                    $booking->flight->departure_time->format('Y-m-d H:i'),
                    $booking->price_paid,
                    $booking->payment_status,
                    $booking->status,
                    $booking->created_at->format('Y-m-d H:i')
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
}
