<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailTransaction;
use App\Models\Settings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmailTransactionController extends Controller
{
    /**
     * Display all email transactions
     */
    public function index()
    {
        $emailTransactions = EmailTransaction::with('user')
            ->orderByDesc('created_at')
            ->paginate(50);

        // Calculate statistics
        $stats = [
            'total' => EmailTransaction::count(),
            'sent' => EmailTransaction::where('status', 'sent')->count(),
            'failed' => EmailTransaction::where('status', 'failed')->count(),
            'pending' => EmailTransaction::where('status', 'pending')->count(),
        ];

        $settings = Settings::where('id', '1')->first();
        $title = 'Email Transactions Management';

        return view('admin.EmailTransactions.email-transactions', compact('emailTransactions', 'stats', 'title'));
    }

    /**
     * Display email transaction details
     */
    public function show($id)
    {
        $emailTransaction = EmailTransaction::with('user')->findOrFail($id);
        $settings = Settings::where('id', '1')->first();
        $title = 'Email Transaction Details - #' . str_pad($emailTransaction->id, 6, '0', STR_PAD_LEFT);

        return view('admin.EmailTransactions.email-transaction-details', compact('emailTransaction', 'settings', 'title'));
    }

    /**
     * Update email transaction status
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $emailTransaction = EmailTransaction::findOrFail($id);

            $request->validate([
                'status' => 'required|in:pending,sent,failed',
                'error_message' => 'nullable|string|max:500'
            ]);

            $emailTransaction->update([
                'status' => $request->status,
                'sent_at' => $request->status === 'sent' ? now() : null,
                'error_message' => $request->status === 'failed' ? $request->error_message : null,
            ]);

            return back()->with('success', 'Email transaction status updated successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update email transaction status: ' . $e->getMessage());
        }
    }    /**
     * Delete email transaction
     */
    public function destroy($id)
    {
        try {
            $emailTransaction = EmailTransaction::findOrFail($id);
            $transactionId = $emailTransaction->transaction_id;
            $emailTransaction->delete();

            return back()->with('success', "Email transaction #{$transactionId} deleted successfully.");

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete email transaction: ' . $e->getMessage());
        }
    }    /**
     * Resend email transaction
     */
    public function resend($id)
    {
        try {
            $emailTransaction = EmailTransaction::findOrFail($id);

            if ($emailTransaction->status === 'sent') {
                return back()->with('warning', 'This email transaction has already been sent successfully.');
            }

            // You can implement the resend logic here
            // For now, just mark as sent
            $emailTransaction->update([
                'status' => 'sent',
                'sent_at' => now(),
                'error_message' => null, // Clear any previous error
            ]);

            return back()->with('success', "Email transaction #{$emailTransaction->transaction_id} resent successfully.");

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to resend email transaction: ' . $e->getMessage());
        }
    }
}
