<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Airport;
use Illuminate\Http\Request;

class AirportController extends Controller
{
    public function __construct()
    {
        $this->middleware('isadmin');
    }

    /**
     * Display a listing of airports.
     */
    public function index()
    {
        $airports = Airport::orderBy('iata_code')->paginate(20);
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Airport Management';
        
        return view('admin.airports.index', compact('airports', 'gradient', 'title'));
    }

    /**
     * Show the form for creating a new airport.
     */
    public function create()
    {
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Add New Airport';
        
        return view('admin.airports.create', compact('gradient', 'title'));
    }

    /**
     * Store a newly created airport.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'iata_code' => 'required|string|size:3|unique:airports,iata_code',
            'icao_code' => 'nullable|string|size:4|unique:airports,icao_code',
            'name' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'timezone' => 'required|string|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'active' => 'boolean',
        ]);

        $validated['iata_code'] = strtoupper($validated['iata_code']);
        if ($validated['icao_code']) {
            $validated['icao_code'] = strtoupper($validated['icao_code']);
        }

        Airport::create($validated);

        return redirect()->route('admin.airports.index')
            ->with('success', 'Airport created successfully!');
    }

    /**
     * Display the specified airport.
     */
    public function show(Airport $airport)
    {
        $airport->load(['originFlights', 'destinationFlights']);
        
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Airport Details';
        
        return view('admin.airports.show', compact('airport', 'gradient', 'title'));
    }

    /**
     * Show the form for editing the specified airport.
     */
    public function edit(Airport $airport)
    {
        // Set gradient for header styling
        $gradient = auth('admin')->user()->dashboard_style == 'light' ? 'primary' : 'dark';
        $title = 'Edit Airport';
        
        return view('admin.airports.edit', compact('airport', 'gradient', 'title'));
    }

    /**
     * Update the specified airport.
     */
    public function update(Request $request, Airport $airport)
    {
        $validated = $request->validate([
            'iata_code' => 'required|string|size:3|unique:airports,iata_code,' . $airport->id,
            'icao_code' => 'nullable|string|size:4|unique:airports,icao_code,' . $airport->id,
            'name' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'timezone' => 'required|string|max:255',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'active' => 'boolean',
        ]);

        $validated['iata_code'] = strtoupper($validated['iata_code']);
        if ($validated['icao_code']) {
            $validated['icao_code'] = strtoupper($validated['icao_code']);
        }

        $airport->update($validated);

        return redirect()->route('admin.airports.index')
            ->with('success', 'Airport updated successfully!');
    }

    /**
     * Remove the specified airport.
     */
    public function destroy(Airport $airport)
    {
        // Check if airport is used by any flights
        if ($airport->originFlights()->count() > 0 || $airport->destinationFlights()->count() > 0) {
            return back()->withErrors(['error' => 'Cannot delete airport that is used by flights.']);
        }

        $airport->delete();

        return redirect()->route('admin.airports.index')
            ->with('success', 'Airport deleted successfully!');
    }
}
