<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Flight;
use Carbon\Carbon;

class UpdateExpiredFlights extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'flights:update-expired';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically update expired flight dates by adding 5 days';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Checking for expired flights...');
        
        // Get all flights with departure time in the past and status scheduled
        $expiredFlights = Flight::where('departure_time', '<', Carbon::now())
            ->whereIn('status', ['scheduled', 'delayed'])
            ->get();
        
        if ($expiredFlights->isEmpty()) {
            $this->info('No expired flights found.');
            return 0;
        }
        
        $this->info("Found {$expiredFlights->count()} expired flight(s). Updating...");
        
        $bar = $this->output->createProgressBar($expiredFlights->count());
        $bar->start();
        
        foreach ($expiredFlights as $flight) {
            $oldDeparture = $flight->departure_time->copy();
            $flightDuration = $flight->departure_time->diffInMinutes($flight->arrival_time);
            
            $newDeparture = $flight->departure_time->copy();
            while ($newDeparture->isPast()) {
                $newDeparture->addDays(5);
            }
            
            $newArrival = $newDeparture->copy()->addMinutes($flightDuration);
            
            $flight->update([
                'departure_time' => $newDeparture,
                'arrival_time' => $newArrival,
            ]);
            
            $this->newLine();
            $this->comment("✓ Flight {$flight->flight_number}: {$oldDeparture->format('Y-m-d H:i')} → {$newDeparture->format('Y-m-d H:i')}");
            
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        $this->info("Successfully updated {$expiredFlights->count()} flight(s)!");
        
        return 0;
    }
}
